#!/usr/bin/env python3
import os, subprocess, tempfile, sys

AFTER_EFFECTS_APP = "Adobe After Effects 2025"

# Paths
PROJECT     = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/ScoringAnimations1-RefTD.aep"
CSV_PATH    = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
COMMON_JSX  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/gl_common.jsxinc"
JSX_BATCH   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_ScoreAnimations1-FG.jsx"

# AE / template
COMP        = "Comp-FG-192"
ANIM_NAME   = "_FG_1_3"
RS_TEMPLATE = "Best Settings"
OM_TEMPLATE = "PNG Sequence"   # any OM containing "png" (case-insensitive) will be accepted

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/FG_Animation_1_192"
PATH_TEMPLATE = "{league}"

# League
LEAGUE = "CFB"   # or "NFL"

# Controls
AE_DEBUG       = "1"
PURGE_BEFORE   = "1"
NO_RENDER      = "0"
QUIT_APP       = "1"

def _write_bridge_jsx(env):
    def esc(s): return s.replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(str(v))+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines) + "\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    print("---- Bridge JSX ----\n" + code + "--------------------", file=sys.stderr)
    return tmp.name

def _osascript(script_text):
    proc = subprocess.run(["osascript", "-s", "o"], input=script_text, text=True, capture_output=True)
    stdout = (proc.stdout or "").strip()
    stderr = (proc.stderr or "").strip()
    print("\n[osascript stdout]\n" + stdout)
    if stderr:
        print("\n[osascript stderr]\n" + stderr, file=sys.stderr)
    proc.check_returncode()
    return stdout, stderr

def _ping_ae(app_name):
    try:
        _osascript(f'tell application "{app_name}" to return name')
        return True
    except subprocess.CalledProcessError:
        return False

def run_after_effects(env):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa_src = f'''
set bridgePosix to "{bridge}"
set bridgeFile to POSIX file bridgePosix
with timeout of 86400 seconds
  tell application "{AFTER_EFFECTS_APP}"
    activate
    try
      set res to DoScriptFile bridgeFile
      return "AERUN:OK:" & (res as text)
    on error errMsg number errNum
      return "AERUN:ERR:" & errNum & ":" & errMsg
    end try
  end tell
end timeout
'''
    _osascript(osa_src)

def make_env():
    return {
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,

        "AE_COMMON_JSX": COMMON_JSX,

        # Layer names
        "AE_SIMPLESTREAK_LAYER": "SimpleStreak",
        "AE_FIELDGOAL_LAYER":    "FG-Text",     # <-- UPDATED default
        "AE_TEAMABBREV_LAYER":   "TeamAbbrev",
        "AE_LOGO_LAYER":         "TeamLogo",

        # Logos
        "AE_LOGO_DIR": "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large",
        "AE_LOGO_PATH_TEMPLATE": "{league}/{abbr}",
        "AE_LOGO_EXTS": "png,jpg,jpeg,svg,ai,psd",

        # Output / templates
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": ANIM_NAME,
        "AE_RS_TEMPLATE": RS_TEMPLATE,
        "AE_OM_TEMPLATE": OM_TEMPLATE,   # PNG OM forced in JSX

        # Controls
        "AE_PURGE_BEFORE_RENDER": PURGE_BEFORE,
        "AE_NO_RENDER": NO_RENDER,
        "AE_QUIT": QUIT_APP,
        "AE_LEAGUE": LEAGUE,
        "AE_DEBUG": AE_DEBUG,
    }

if __name__ == "__main__":
    print("== sanity checks ==")
    assert os.path.isfile(JSX_BATCH), f"JSX not found: {JSX_BATCH}"
    assert os.path.isfile(COMMON_JSX), f"Common lib not found: {COMMON_JSX}"
    assert os.path.isfile(PROJECT),   f"Project not found: {PROJECT}"
    assert os.path.isfile(CSV_PATH),  f"CSV not found: {CSV_PATH}"

    if not _ping_ae(AFTER_EFFECTS_APP):
        print(f'!! Could not talk to "{AFTER_EFFECTS_APP}". Check the exact app name in /Applications', file=sys.stderr)
        raise SystemExit(2)

    print("\n== launching AE with script ==")
    run_after_effects(make_env())
    print("\nDone.")
